/***************************************************************************
 *
 * Copyright 2010,2011 BMW Car IT GmbH
 * Copyright (C) 2012 DENSO CORPORATION and Robert Bosch Car Multimedia Gmbh
 *
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ****************************************************************************/
#include "ilm_client.h"
#include "ilm_control.h"
#include "LayerScene.h"
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <unistd.h>
#include <getopt.h>
#include <string>
#include <string.h>

t_ilm_uint g_screenWidth = 800;
t_ilm_uint g_screenHeight = 480;
t_ilm_uint g_surfaceWidth = 800;
t_ilm_uint g_surfaceHeight = 480;

t_ilm_layer g_layer_baseid = LAYER_EXAMPLE_GLES_APPLICATIONS + 1;
t_ilm_layer* gp_layer_ids = NULL;
t_ilm_layer g_surface_id = SURFACE_EXAMPLE_EGLX11_APPLICATION;

ilmErrorTypes createLayer(t_ilm_layer* layerid) 
{   
    ilmErrorTypes ans = ILM_SUCCESS;

    printf("create layer %u\n", *layerid);
    ans = ilm_layerCreateWithDimension( layerid, g_surfaceWidth, g_surfaceHeight );
    if (ILM_SUCCESS != ans)
    {
        printf("Failed:ilm_layerCreate\n");
        return ans;
    }

    ans = ilm_layerSetDestinationRectangle(*layerid, 0, 0, g_surfaceWidth, g_surfaceHeight);
    if (ILM_SUCCESS != ans)
    {
        printf("Failed:ilm_layerSetDestinationRectangle\n");
        return ans;
    }

    ans = ilm_layerSetSourceRectangle(*layerid, 0, 0, g_surfaceWidth, g_surfaceHeight);
    if (ILM_SUCCESS != ans)
    {
        printf("Failed:ilm_layerSetSourceRectangle\n");
        return ans;
    }

    ans = ilm_layerSetVisibility(*layerid, ILM_TRUE);
    if (ILM_SUCCESS != ans)
    {
        printf("Failed:ilm_layerSetVisibility\n");
        return ans;
    }

    ans = ilm_layerSetOpacity(*layerid, 1.0);
    if (ILM_SUCCESS != ans)
    {
        printf("Failed:ilm_layerSetOpacity\n");
        return ans;
    }

    ans = ilm_commitChanges();
    if (ILM_SUCCESS != ans)
    {
        printf("Failed:ilm_commitChanges\n");
        return ans;
    }

    return ILM_SUCCESS;
}

void removeLayer(t_ilm_layer layerid)
{  
    ilm_layerRemove(layerid);
    ilm_commitChanges();
}

void getSurfaceDestSize(t_ilm_surface surfaceid, t_ilm_uint* width, t_ilm_uint* height)
{
    struct ilmSurfaceProperties surfaceProperties;
    ilm_getPropertiesOfSurface(surfaceid, &surfaceProperties);
    *width = surfaceProperties.destWidth;
    *height = surfaceProperties.destHeight;
}

t_ilm_uint g_screenNum = 0;
t_ilm_uint* g_screenIDs = NULL;

t_ilm_int g_moveX = 0;
t_ilm_int g_moveY = 0;
int g_moveDirX = 0; // 0:right, 1:left
int g_moveDirY = 0; // 0:down, 1:up

void moveLayer()
{
    int cntX = 0;
    int cntY = 0;
    int screenXnum = ((g_screenNum + 1) >> 1);
    int screenYnum = 2;
    if (1 == g_screenNum)
    {
        screenYnum = 1;
    }

    if (0 == g_moveDirX)
    {
        g_moveX+=1;
        if ((g_moveX + g_surfaceWidth) >= (g_surfaceWidth * screenXnum))
        {
            g_moveDirX = 1;
            g_moveX-=1;
        }
    }
    else
    {
        g_moveX-=1;
        if (g_moveX < 0)
        {
            g_moveDirX = 0;
            g_moveX+=1;
        }
    }
    if (0 == g_moveDirY)
    {
        g_moveY+=2;
        if ((g_moveY + g_surfaceHeight) >= (g_surfaceHeight * screenYnum))
        {
            g_moveDirY = 1;
            g_moveY-=2;
        }
    }
    else
    {
        g_moveY-=2;
        if (g_moveY < 0)
        {
            g_moveDirY = 0;
            g_moveY+=2;
        }
    }

    int orgSrcX = 0;
    int orgSrcY = 0;
    int orgDstX = 0;
    int orgDstY = 0;
    int width = 0;
    int height = 0;
    for (cntY = 0; cntY < screenYnum; cntY++)
    {
        for (cntX = 0; cntX < screenXnum; cntX++)
        {
            orgSrcX = 0;
            orgSrcY = 0;
            orgDstX = 0;
            orgDstY = 0;
            width = 0;
            height = 0;
            if ((cntX + cntY * screenXnum) == g_screenNum)
            {
                ilm_layerSetVisibility(gp_layer_ids[cntX + cntY * screenXnum], ILM_FALSE);
                break;
            }

            orgDstX = g_moveX - (cntX * g_screenWidth);
            orgDstY = g_moveY - (cntY * g_screenHeight);
            if ((0 >= (orgDstX + g_surfaceWidth)) || (0 >= (orgDstY + g_surfaceHeight))
             || ((int)g_surfaceWidth <= orgDstX) || ((int)g_surfaceHeight <= orgDstY))
            {
                ilm_layerSetVisibility(gp_layer_ids[cntX + cntY * screenXnum], ILM_FALSE);
                continue;
            }

            if ((orgDstX + g_surfaceWidth) >= (cntX * g_screenWidth))
            {
                width = g_surfaceWidth - (orgDstX - (cntX * g_screenWidth));
            }
            if ((orgDstY + g_surfaceHeight) >= (cntY * g_screenHeight))
            {
                height = g_surfaceHeight - (orgDstY - (cntY * g_screenHeight));
            }
            if (0 > orgDstX)
            {
                width = g_surfaceWidth + orgDstX;
                orgSrcX = -orgDstX;
                orgDstX = 0;
            }
            if (0 > orgDstY)
            {
                height = g_surfaceHeight + orgDstY;
                orgSrcY = -orgDstY;
                orgDstY = 0;
            }

            ilm_layerSetVisibility(gp_layer_ids[cntX + cntY * screenXnum], ILM_TRUE);
            ilm_layerSetSourceRectangle(gp_layer_ids[cntX + cntY * screenXnum], orgSrcX, orgSrcY, width, height);
            ilm_layerSetDestinationRectangle(gp_layer_ids[cntX + cntY * screenXnum], orgDstX, orgDstY, width, height);
        }
    }

    ilm_commitChanges();
}

int main(int argc, char **argv)
{
    ilmErrorTypes result = ILM_FAILED;
    int cnt = 0;
    int child = 0;
    int child_pid = 0;
    char str_surface_id[10] = "";
    char str_layer_id[10] = "";

    if (argc == 2)
    {
        g_surface_id = atoi(argv[1]);
    }

    do
    {
        result = ilm_init();
        if (ILM_SUCCESS != result)
        {
            break;
        }

        result = ilm_getScreenIDs(&g_screenNum, &g_screenIDs);
        if (ILM_SUCCESS != result)
        {
            break;
        }

        printf("The number of screenIDs: %d\n", g_screenNum);
        gp_layer_ids = (t_ilm_layer*)malloc(sizeof(t_ilm_layer) * g_screenNum);

        if (NULL == gp_layer_ids)
        {
            printf("Failed: alloc array of ids\n");
            break;
        }

        printf("Available display outputs:\n");
        for (cnt = 0; cnt < (int)g_screenNum; cnt++)
        {
           if( g_screenIDs[cnt] == ILM_DISPLAY_HDMI){
               ilm_getScreenResolution(ILM_DISPLAY_HDMI, &g_screenWidth, &g_screenHeight);
               printf("\tHDMI with screen id:%d, resolution:%dx%d\n",g_screenIDs[cnt],g_screenWidth, g_screenHeight);
           }
           else if( g_screenIDs[cnt] == ILM_DISPLAY_LVDS0){
               ilm_getScreenResolution(ILM_DISPLAY_LVDS0, &g_screenWidth, &g_screenHeight);
               printf("\tLVDS0 with screen id:%d, resolution:%dx%d\n",g_screenIDs[cnt],g_screenWidth, g_screenHeight);
           }
           else if( g_screenIDs[cnt] == ILM_DISPLAY_LVDS1){
               ilm_getScreenResolution(ILM_DISPLAY_LVDS1, &g_screenWidth, &g_screenHeight);
               printf("\tLVDS1 with screen id:%d, resolution:%dx%d\n",g_screenIDs[cnt],g_screenWidth, g_screenHeight);
           }
           else if( g_screenIDs[cnt] == ILM_DISPLAY_PARALLEL){
               ilm_getScreenResolution(ILM_DISPLAY_PARALLEL, &g_screenWidth, &g_screenHeight);
               printf("\tPARALLEL on screen id:%d, resolution:%dx%d\n",g_screenIDs[cnt],g_screenWidth, g_screenHeight);
           }
        }
        gp_layer_ids[0] = (t_ilm_layer)g_layer_baseid;
        result = createLayer(&gp_layer_ids[0]);
        if (ILM_SUCCESS != result)
        {
            break;
        }

        if ((child = fork()) == 0)
        {
            child_pid = getpid();
            printf("Entering the child %d process \n", child_pid);
            sprintf(str_surface_id, "%d", g_surface_id);
            sprintf(str_layer_id, "%d", gp_layer_ids[0]);
            execl("/usr/bin/EGLWLMockNavigation", "EGLWLMockNavigation",
                    "-surface", str_surface_id, "-layer",str_layer_id, NULL);

            printf("Execv failed to execute, Hence exit this child\n");
            exit(1);
        }
        else if (child < 0)
        {
            printf("Creation of child failed \n");
        }
        sleep(2);/*wait until application starts and create the surface*/
        printf("add surface to layer\n");
        /*surface is created by the external application*/
        result = ilm_layerAddSurface(gp_layer_ids[0], g_surface_id);
        if (ILM_SUCCESS != result)
        {
            printf("Failed: ilm_layerAddSurface\n");
            break;
        }

        result = ilm_commitChanges();
        if (ILM_SUCCESS != result)
        {
            printf("Failed(commit): ilm_commitChanges\n");
            break;
        }

        /*Display the layer on HDMI only*/
        printf("Display the layer on HDMI\n");
        result = ilm_displaySetRenderOrder(ILM_DISPLAY_HDMI, &gp_layer_ids[0], 1);
        if (ILM_SUCCESS != result)
        {
            printf("Failed: ilm_displaySetRenderOrder\n");
            break;
        }
        result = ilm_commitChanges();
        if (ILM_SUCCESS != result)
        {
            printf("Failed(commit): ilm_commitChanges\n");
            break;
        }
        sleep(3);
        /*Display the layer on HDMI and LVDS0*/
        printf("Display the layer on HDMI and LVDS0\n");
        result = ilm_displaySetRenderOrder(ILM_DISPLAY_LVDS0, &gp_layer_ids[0], 1);
        if (ILM_SUCCESS != result)
        {
            printf("Failed: ilm_displaySetRenderOrder\n");
            break;
        }
        result = ilm_commitChanges();
        if (ILM_SUCCESS != result)
        {
            printf("Failed(commit): ilm_commitChanges\n");
            break;
        }
        sleep(3);

        /*Display the layer on HDMI and LVDS1(if available)*/
        printf("Display the layer on HDMI and LVDS1 (if available)\n");
        /*remove layer from LVDS0*/
        result = ilm_displaySetRenderOrder(ILM_DISPLAY_LVDS0, &gp_layer_ids[0], 0);
        if (ILM_SUCCESS != result)
        {
            printf("Failed: ilm_displaySetRenderOrder\n");
            break;
        }
        result = ilm_displaySetRenderOrder(ILM_DISPLAY_LVDS1, &gp_layer_ids[0], 1);
        if (ILM_SUCCESS != result)
        {
            printf("Failed: ilm_displaySetRenderOrder\n");
            break;
        }
        result = ilm_commitChanges();
        if (ILM_SUCCESS != result)
        {
            printf("Failed(commit): ilm_commitChanges\n");
            break;
        }
        sleep(3);
        /*Display the layer on HDMI and LVDS1*/
        printf("Display the layer on HDMI, LVDS0 and LVDS1\n");
        /*remove layer from LVDS0*/
        result = ilm_displaySetRenderOrder(ILM_DISPLAY_LVDS0, &gp_layer_ids[0], 1);
        if (ILM_SUCCESS != result)
        {
           printf("Failed: ilm_displaySetRenderOrder\n");
           break;
        }
        result = ilm_commitChanges();
        if (ILM_SUCCESS != result)
        {
           printf("Failed(commit): ilm_commitChanges\n");
           break;
        }

    } while(0);
    printf("Changes on layer should be visible on HDMI,LVDS0 and LVDS1\n");

    while (1)
    {
        moveLayer();
        usleep(100000);
    }

    for (cnt = 0; cnt < (int)g_screenNum; cnt++)
    {
        removeLayer(gp_layer_ids[cnt]);
    }

    if (NULL != gp_layer_ids)
    {
        free(gp_layer_ids);
        gp_layer_ids = NULL;
    }
    ilm_destroy();
}
